package main

import (
	"fmt"
	"math/rand"
	"time"
)

type Result string

type Search func(query string) Result

func fakeServe(kind, query string) Result {
	time.Sleep(time.Duration(rand.Intn(100)) * time.Millisecond)
	return Result(fmt.Sprintf("%s result for %q\n", kind, query))
}

// Replication in order to avoid timeouts
func First(query string, replicas ...string) Result {
	c := make(chan Result)
	searchReplica := func(i int) { c <- fakeServe(replicas[i], query) }
	for i := range replicas {
		go searchReplica(i)
	}
	return <-c
}

// Main search function
func Youtube(query string) (results []Result) {
	c := make(chan Result)

	go func() { c <- First(query, "Video1", "Video2") }()
	go func() { c <- First(query, "Suggestion1", "Suggestion2") }()
	go func() { c <- First(query, "Ads1", "Ads2") }()

	timeout := time.After(80 * time.Millisecond)

	for i := 0; i < 3; i++ {
		select {
		case result := <-c:
			results = append(results, result)
		case <-timeout:
			fmt.Println("timed out.")
			return
		}
	}
	return results
}

func main() {
	rand.Seed(time.Now().UnixNano())
	start := time.Now()
	results := Youtube("cats")
	elapsed := time.Since(start)
	fmt.Println(results)
	fmt.Println(elapsed)
}
